// ZEN Study Plus Welcome Page JavaScript
// CSP-compliant implementation with enhanced features

class ZenWelcome {
    constructor() {
        this.isLoading = true;
        this.animationQueue = [];
        this.observers = [];
        this.particles = [];
        this.init();
    }

    init() {
        this.setupLoading();
        this.setupScrollAnimations();
        this.setupNavigationEffects();
        this.setupParticleSystem();
        this.setupEventListeners();
        this.initializeFeatureCards();
    }

    // Loading screen management
    setupLoading() {
        window.addEventListener('load', () => {
            setTimeout(() => {
                const loadingScreen = document.getElementById('loading');
                if (loadingScreen) {
                    loadingScreen.classList.add('hidden');
                }
                this.isLoading = false;
                this.startMainAnimations();
            }, 1000);
        });
    }

    // Initialize scroll-triggered animations
    setupScrollAnimations() {
        const observerOptions = {
            threshold: 0.2,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    this.animateElement(entry.target);
                    observer.unobserve(entry.target);
                }
            });
        }, observerOptions);

        this.observers.push(observer);

        // Observe feature cards
        window.addEventListener('load', () => {
            const featureCards = document.querySelectorAll('.feature-card');
            const sections = document.querySelectorAll('.hero, .features, .cta');
            
            featureCards.forEach(card => observer.observe(card));
            sections.forEach(section => observer.observe(section));
        });
    }

    // Setup navigation smooth scrolling and effects
    setupNavigationEffects() {
        // Smooth scroll for navigation links
        document.addEventListener('click', (e) => {
            if (e.target.matches('a[href^="#"]')) {
                e.preventDefault();
                const targetId = e.target.getAttribute('href').substring(1);
                const targetElement = document.getElementById(targetId);
                
                if (targetElement) {
                    targetElement.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            }
        });

        // Header scroll effect
        let lastScrollY = window.scrollY;
        window.addEventListener('scroll', () => {
            const currentScrollY = window.scrollY;
            const header = document.querySelector('header');
            
            if (header) {
                if (currentScrollY > 100) {
                    header.classList.add('scrolled');
                } else {
                    header.classList.remove('scrolled');
                }
            }
            
            lastScrollY = currentScrollY;
        });
    }

    // Particle system for background animation
    setupParticleSystem() {
        const canvas = document.getElementById('particles');
        if (!canvas) return;

        const ctx = canvas.getContext('2d');
        
        const resizeCanvas = () => {
            canvas.width = window.innerWidth;
            canvas.height = window.innerHeight;
        };
        
        resizeCanvas();
        window.addEventListener('resize', resizeCanvas);

        // Create particles
        for (let i = 0; i < 50; i++) {
            this.particles.push({
                x: Math.random() * canvas.width,
                y: Math.random() * canvas.height,
                vx: (Math.random() - 0.5) * 0.5,
                vy: (Math.random() - 0.5) * 0.5,
                size: Math.random() * 2 + 1,
                opacity: Math.random() * 0.5 + 0.2
            });
        }

        const animate = () => {
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            
            this.particles.forEach(particle => {
                particle.x += particle.vx;
                particle.y += particle.vy;
                
                if (particle.x < 0 || particle.x > canvas.width) particle.vx *= -1;
                if (particle.y < 0 || particle.y > canvas.height) particle.vy *= -1;
                
                ctx.beginPath();
                ctx.arc(particle.x, particle.y, particle.size, 0, Math.PI * 2);
                ctx.fillStyle = `rgba(74, 144, 226, ${particle.opacity})`;
                ctx.fill();
            });
            
            requestAnimationFrame(animate);
        };
        
        animate();
    }

    // Event listeners for interactive elements
    setupEventListeners() {
        // Opening settings page
        document.querySelectorAll('#openSettingsBtn, #openSettingsBtn2, #headerSettingsBtn').forEach(btn => {
            if (btn) {
                btn.addEventListener('click', (e) => {
                    e.preventDefault();
                    // Open settings directly
                    this.openSettings();
                });
            }
        });

        // Show quick guide dialog
        const showQuickGuideBtn = document.getElementById('showQuickGuideBtn');
        if (showQuickGuideBtn) {
            showQuickGuideBtn.addEventListener('click', (e) => {
                e.preventDefault();
                this.showQuickGuideDialog();
            });
        }
        
        // Account section related events
        const accountSection = document.querySelector('#account');
        if (accountSection) {
            // Check if user is logged in
            chrome.storage.sync.get(['userProfile'], (result) => {
                if (result.userProfile) {
                    // User is logged in
                    this.showLoggedInState(result.userProfile);
                }
            });
        }
    }

    // Display logged-in state in account section
    showLoggedInState(profile) {
        const accountSection = document.querySelector('#account');
        if (!accountSection) return;
        
        // Add login status to first feature card
        const firstCard = accountSection.querySelector('.feature-card');
        if (firstCard) {
            const loginStatus = document.createElement('div');
            loginStatus.className = 'login-status';
            loginStatus.innerHTML = `
                <div class="login-badge success">
                    <span class="icon">✓</span>
                    <span class="text">ログイン中</span>
                </div>
                <p class="login-info">アカウント: ${profile.email || profile.username || 'ユーザー'}</p>
            `;
            
            // Add styles
            const style = document.createElement('style');
            style.textContent = `
                .login-status {
                    margin-top: 20px;
                    padding-top: 15px;
                    border-top: 1px solid var(--light-gray);
                }
                .login-badge {
                    display: inline-flex;
                    align-items: center;
                    padding: 5px 12px;
                    border-radius: 15px;
                    font-size: 0.85rem;
                    font-weight: 500;
                    background: var(--success);
                    color: white;
                    margin-bottom: 10px;
                }
                .login-badge .icon {
                    margin-right: 5px;
                }
                .login-info {
                    font-size: 0.9rem;
                    color: var(--gray);
                }
            `;
            document.head.appendChild(style);
            
            firstCard.appendChild(loginStatus);
        }
    }

    // Initialize feature cards
    initializeFeatureCards() {
        const featureCards = document.querySelectorAll('.feature-card');
        
        featureCards.forEach((card, index) => {
            // Add delay for staggered animation
            setTimeout(() => {
                card.classList.add('visible');
            }, 100 * index);
            
            // Add hover effect
            card.addEventListener('mouseenter', () => {
                card.style.transform = 'translateY(-10px)';
                card.style.boxShadow = '0 20px 40px rgba(0, 0, 0, 0.1)';
            });
            
            card.addEventListener('mouseleave', () => {
                card.style.transform = 'translateY(0)';
                card.style.boxShadow = '0 10px 30px rgba(0, 0, 0, 0.05)';
            });
        });
    }

    // Animation methods
    animateElement(element) {
        element.classList.add('animate-in');
        
        if (element.classList.contains('feature-card')) {
            element.classList.add('visible');
            const icon = element.querySelector('.feature-icon');
            if (icon) {
                setTimeout(() => {
                    icon.style.transform = 'scale(1.1)';
                    setTimeout(() => {
                        icon.style.transform = 'scale(1)';
                    }, 200);
                }, 300);
            }
        }
    }

    startMainAnimations() {
        const heroContent = document.querySelector('.hero-content');
        const installBadge = document.querySelector('.install-complete');
        
        if (heroContent) {
            heroContent.classList.add('fade-in');
        }
        
        if (installBadge) {
            setTimeout(() => {
                installBadge.classList.add('bounce-in');
            }, 500);
        }
    }

    // Event handlers
    openSettings() {
        // Try to open popup as a real popup, not a new tab
        if (typeof chrome !== 'undefined' && chrome.runtime) {
            // First try with the action API (Manifest V3)
            if (chrome.action && chrome.action.openPopup) {
                try {
                    chrome.action.openPopup(() => {
                        if (chrome.runtime.lastError) {
                            console.error('ポップアップを開けませんでした:', chrome.runtime.lastError);
                            this.tryAlternativeOpenMethod();
                        }
                    });
                    return;
                } catch (error) {
                    console.error('chrome.action.openPopup エラー:', error);
                    // Fall through to next method
                }
            }
            
            // Then try sending a message to background script
            chrome.runtime.sendMessage({ action: "OPEN_POPUP" }, (response) => {
                if (chrome.runtime.lastError || !response || !response.success) {
                    this.showNotification('設定を開くには、拡張機能アイコンをクリックしてください', 'info');
                }
            });
        } else {
            this.showNotification('設定を開くには、ツールバーの拡張機能アイコンをクリックしてください。', 'info');
        }
    }
    
    tryAlternativeOpenMethod() {
        // Last resort - create a small floating window that simulates a popup
        try {
            const width = 400;
            const height = 600;
            const left = (window.screen.width / 2) - (width / 2);
            const top = (window.screen.height / 2) - (height / 2);
            
            const popupURL = chrome.runtime.getURL('popup.html');
            
            chrome.windows.create({
                url: popupURL,
                type: 'popup',
                width: width,
                height: height,
                left: Math.round(left),
                top: Math.round(top),
                focused: true
            }, (window) => {
                if (chrome.runtime.lastError) {
                    console.error('ポップアップウィンドウを開けませんでした:', chrome.runtime.lastError);
                    this.showLastResortNotification();
                }
            });
        } catch (error) {
            console.error('ポップアップウィンドウ作成エラー:', error);
            this.showLastResortNotification();
        }
    }
    
    showLastResortNotification() {
        // Fallback notification with instructions
        this.showNotification('設定を開くには、拡張機能アイコンをクリックしてください', 'info');
        
        // Highlight the extension icon location
        const iconLocationMsg = document.createElement('div');
        iconLocationMsg.innerHTML = `
            <div style="position: fixed; top: 15px; right: 20px; background: rgba(0,0,0,0.8); color: white; padding: 12px 15px; border-radius: 8px; z-index: 9999; max-width: 300px;">
                <p style="margin: 0 0 8px 0; font-weight: bold;">拡張機能アイコンの場所</p>
                <p style="margin: 0;">ブラウザ右上のパズルピースアイコンをクリックし、ZEN Study Plusを選択してください。</p>
            </div>
        `;
        document.body.appendChild(iconLocationMsg);
        
        setTimeout(() => {
            if (iconLocationMsg.parentNode) {
                iconLocationMsg.parentNode.removeChild(iconLocationMsg);
            }
        }, 8000);
    }

    handleFeatureToggle(button) {
        const isActive = button.classList.contains('active');
        button.classList.toggle('active');
        
        const featureName = button.getAttribute('data-feature') || 'Unknown';
        const status = isActive ? '無効' : '有効';
        
        this.showNotification(`${featureName}機能を${status}にしました`, 'success');
    }

    handleActionButton(button) {
        const action = button.getAttribute('data-action');
        
        switch(action) {
            case 'tutorial':
                this.startTutorial();
                break;
            case 'import':
                this.importSettings();
                break;
            case 'export':
                this.exportSettings();
                break;
            default:
                // 未知のアクションは無視
        }
    }

    handleGetStarted() {
        // Scroll to features section
        const featuresSection = document.getElementById('features');
        if (featuresSection) {
            featuresSection.scrollIntoView({ behavior: 'smooth' });
        }
        
        this.showNotification('機能の設定を確認して、学習を始めましょう！', 'info');
    }

    openSupport() {
        // Open support page in new tab
        window.open('https://github.com/yoima-jp/zen-Plus-web/issues', '_blank');
    }

    showQuickGuide() {
        // Show quick guide notification and scroll to features
        this.showNotification('クイックガイド: 下の機能セクションで各機能の詳細を確認できます', 'info');
        const featuresSection = document.getElementById('features');
        if (featuresSection) {
            setTimeout(() => {
                featuresSection.scrollIntoView({ behavior: 'smooth' });
            }, 1000);
        }
    }

    // Show quick guide dialog
    showQuickGuideDialog() {
        // Create modal overlay
        const overlay = document.createElement('div');
        overlay.className = 'modal-overlay';
        Object.assign(overlay.style, {
            position: 'fixed',
            top: '0',
            left: '0',
            width: '100%',
            height: '100%',
            backgroundColor: 'rgba(0, 0, 0, 0.7)',
            display: 'flex',
            justifyContent: 'center',
            alignItems: 'center',
            zIndex: '9999',
            opacity: '0',
            transition: 'opacity 0.3s ease'
        });

        // Create modal content
        const modal = document.createElement('div');
        modal.className = 'modal-content';
        Object.assign(modal.style, {
            backgroundColor: 'white',
            borderRadius: '10px',
            padding: '30px',
            maxWidth: '600px',
            width: '90%',
            maxHeight: '80vh',
            overflow: 'auto',
            boxShadow: '0 10px 30px rgba(0, 0, 0, 0.2)',
            transform: 'translateY(20px)',
            transition: 'transform 0.3s ease',
            position: 'relative'
        });

        // Add close button
        const closeBtn = document.createElement('button');
        closeBtn.innerText = '×';
        Object.assign(closeBtn.style, {
            position: 'absolute',
            top: '10px',
            right: '15px',
            border: 'none',
            background: 'transparent',
            fontSize: '24px',
            cursor: 'pointer',
            color: 'var(--gray)'
        });
        closeBtn.addEventListener('click', () => {
            overlay.style.opacity = '0';
            modal.style.transform = 'translateY(20px)';
            setTimeout(() => {
                document.body.removeChild(overlay);
            }, 300);
        });

        // Populate content
        modal.innerHTML = `
            <h2 style="margin-bottom: 20px; color: var(--primary);">ZEN Study Plus クイックガイド</h2>
            <div class="guide-steps">
                <div class="guide-step">
                    <span class="step-number">1</span>
                    <div class="step-content">
                        <h3>インストール完了</h3>
                        <p>拡張機能はすでにインストール済みで、ZEN Studyでご利用いただける状態です。</p>
                    </div>
                </div>
                <div class="guide-step">
                    <span class="step-number">2</span>
                    <div class="step-content">
                        <h3>N高等学校・S高等学校のサイトにアクセス</h3>
                        <p>ZEN Studyにログインすると、拡張機能が自動的に機能し始めます。</p>
                    </div>
                </div>
                <div class="guide-step">
                    <span class="step-number">3</span>
                    <div class="step-content">
                        <h3>設定をカスタマイズ</h3>
                        <p>ブラウザツールバーの拡張機能アイコンをクリックして、設定パネルを開きます。お好みに合わせて機能の有効/無効を切り替えられます。</p>
                    </div>
                </div>
                <div class="guide-step">
                    <span class="step-number">4</span>
                    <div class="step-content">
                        <h3>学習を開始</h3>
                        <p>チャプターページでは、残りの学習量が自動的に計算され、画面上部に表示されます。これで効率的な学習計画が立てられます。</p>
                    </div>
                </div>
                <div class="guide-step">
                    <span class="step-number">5</span>
                    <div class="step-content">
                        <h3>アカウント作成（任意）</h3>
                        <p>複数のデバイスで学習履歴を同期したい場合は、設定画面からアカウントを作成してください。</p>
                    </div>
                </div>
            </div>
            <div class="guide-footer" style="margin-top: 30px; text-align: center;">
                <button id="closeGuideBtn" class="btn primary" style="padding: 10px 25px;">理解しました</button>
            </div>
        `;

        // Add guide step styles
        const style = document.createElement('style');
        style.textContent = `
            .guide-steps {
                display: flex;
                flex-direction: column;
                gap: 20px;
            }
            .guide-step {
                display: flex;
                gap: 15px;
            }
            .step-number {
                background: linear-gradient(135deg, var(--primary), var(--secondary));
                color: white;
                width: 30px;
                height: 30px;
                display: flex;
                align-items: center;
                justify-content: center;
                border-radius: 50%;
                font-weight: bold;
            }
            .step-content h3 {
                margin-bottom: 5px;
                color: var(--dark);
            }
            .step-content p {
                color: var(--gray);
                font-size: 0.95rem;
            }
        `;
        document.head.appendChild(style);

        // Add close button to modal
        modal.appendChild(closeBtn);

        // Add close functionality for the OK button
        overlay.appendChild(modal);
        document.body.appendChild(overlay);

        // Animate opening
        setTimeout(() => {
            overlay.style.opacity = '1';
            modal.style.transform = 'translateY(0)';
        }, 10);

        // Add event listener to the "理解しました" button
        setTimeout(() => {
            const closeGuideBtn = document.getElementById('closeGuideBtn');
            if (closeGuideBtn) {
                closeGuideBtn.addEventListener('click', () => {
                    overlay.style.opacity = '0';
                    modal.style.transform = 'translateY(20px)';
                    setTimeout(() => {
                        document.body.removeChild(overlay);
                    }, 300);
                });
            }
        }, 100);
    }

    // Utility methods
    startTutorial() {
        this.showNotification('チュートリアルを開始します...', 'info');
        // Tutorial logic would go here
    }

    importSettings() {
        const input = document.createElement('input');
        input.type = 'file';
        input.accept = '.json';
        input.addEventListener('change', (e) => {
            const file = e.target.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = (event) => {
                    try {
                        const settings = JSON.parse(event.target.result);
                        this.showNotification('設定をインポートしました', 'success');
                    } catch (error) {
                        this.showNotification('設定ファイルの読み込みに失敗しました', 'error');
                    }
                };
                reader.readAsText(file);
            }
        });
        input.click();
    }

    exportSettings() {
        const settings = {
            version: '1.0.0',
            timestamp: new Date().toISOString(),
            features: {
                focusMode: true,
                timeTracking: true,
                websiteBlocking: false
            }
        };
        
        const blob = new Blob([JSON.stringify(settings, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'zen-study-plus-settings.json';
        a.click();
        URL.revokeObjectURL(url);
        
        this.showNotification('設定をエクスポートしました', 'success');
    }

    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification ${type}`;
        notification.textContent = message;
        
        // Style the notification
        Object.assign(notification.style, {
            position: 'fixed',
            top: '20px',
            right: '20px',
            padding: '12px 20px',
            borderRadius: '8px',
            color: 'white',
            fontSize: '14px',
            fontWeight: '500',
            zIndex: '10000',
            transition: 'all 0.3s ease',
            transform: 'translateX(400px)',
            opacity: '0'
        });
        
        // Set background color based on type
        switch(type) {
            case 'success':
                notification.style.backgroundColor = '#10b981';
                break;
            case 'error':
                notification.style.backgroundColor = '#ef4444';
                break;
            case 'warning':
                notification.style.backgroundColor = '#f59e0b';
                break;
            default:
                notification.style.backgroundColor = '#3b82f6';
        }
        
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.style.transform = 'translateX(0)';
            notification.style.opacity = '1';
        }, 100);
        
        // Remove after delay
        setTimeout(() => {
            notification.style.transform = 'translateX(400px)';
            notification.style.opacity = '0';
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 3000);
    }

    // Cleanup method
    destroy() {
        this.observers.forEach(observer => observer.disconnect());
        this.observers = [];
        this.particles = [];
    }
}

// Initialize the welcome page
document.addEventListener('DOMContentLoaded', () => {
    window.zenWelcome = new ZenWelcome();
});

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    if (window.zenWelcome) {
        window.zenWelcome.destroy();
    }
});
